
import { ActionButton } from '/sd:action-button.js'; 
import { Point } from '/sd:point.js';
/**
 * class to create instance of Button, which displays a point value. It allows for define button with label text inside div container, and to invoke action onclick div container. It creates effect of clicked and non clicked button.
 */
export class ActionButtonWithValue extends ActionButton {	
	/**
	 * Constructor
	 * @param    {jQuery} xmlObject    		Data loaded from XML
	 * @param    {String} parentId    		HTML ID of the object, which has to be a parent of constructing object
	 * @param	 {String} clickAction		This value is insert inside of onclck="" attribute of HTML button 
	 * @param    {Integer} itemNumber   	Number of object constructing item - if it is null, it means, that item is not numerable
	 * @param    {String} pointMainClass    Name of the CSS style's class
	 * @param    {String} objId    			ID of creating object - if itemNumber is not null, then itemNumber will be suffix of the ID
	 * @param    {Collection} valueAttributes 	Collection of event value attributes {"unit": "", "scale": 1.0, "floatSeparator": "dot", "enum": "", "falseText": "", "trueText": ""}
	 * @param 	 {jQuery} enumsDefXmlObject     Data loaded from XML - accept enums definition part only
	 */
	constructor(xmlObject, parentId, clickAction, itemNumber = null, pointMainClass = "ActionButton", objId = null, pointRefreshFrequency, holdingRegister=null, bitNumber="All", valueAttributes, enumsDefXmlObject) {
		super(xmlObject, parentId, clickAction, itemNumber, pointMainClass, objId, true);
		this.point = new Point("unknown", pointRefreshFrequency, holdingRegister, valueAttributes.scale, valueAttributes.unit, valueAttributes.floatSeparator, valueAttributes.enum, enumsDefXmlObject, bitNumber, valueAttributes.falseText, valueAttributes.trueText);
	}

	/**
	 * Gets bit number of boolean value
	 * @returns {String}	"All" or bit number
	 */
	getBitNumber() {
		return this.point.getBitNumber();
	}

	/**
	 * Gets decimal places
	 * @returns {Integer}	Day number
	 */
	getDecimlPlaces() {
		return this.point.getDecimlPlaces();
	}

	/**
	 * Gets false text value
	 * @returns {String}	False text of the point
	 */
	getFalseText() {
		return this.point.getFalseText();
	}
	
	/**
	 * Gets label text of the field
	 * @returns {String}	Label text
	 */
	getLabelText() {
		return "";
	}
	
	/**
	 * Gets starting Modbus register address of point
	 * @returns {Integer}		Register address
	 */
	getModbusRegisterAddress() {
		return this.point.getModbusRegisterAddress();
	}

	/**
	 * Gets name of pointRefreshFrequency
	 * @returns {String}	Name of point refresh frequency tunning
	 */
	getPointRefreshFrequency() {
		return this.point.getPointRefreshFrequency();
	}
	
	/**
	 * Gets type of the point
	 * @returns {String}	Type of the point ("numeric", "enum", "boolean" or "unknown")
	 */
	getPointType() {
		return this.point.getPointType();
	}
	
	/**
	 * Gets scale of the numeric point
	 * @returns {Float}		Scale value
	 */
	getScale() {
		return this.point.getScale();
	}

	/**
	 * Gets current state of the event
	 * @returns {Boolean}	Current state of the event
	 */
	getState() {
		return this.point.getState();
	}
	
	/**
	 * Gets true text value
	 * @returns {String}	True text of the point
	 */
	getTrueText() {
		return this.point.getTrueText();
	}
	
	/**
	 * Gets unit of the numeric point
	 * @returns {String}	Unit of the numeric point
	 */
	getUnit() {
		return this.point.getUnit();
	}
	
	/**
	 * Gets current value of the event field
	 * @returns {Object} 	Value of the numeric point
	 */
	getValue() {
		return this.point.getValue();
	}
	
	/**
	 * Gets rescaled value of the numeric point for Modbus TCP
	 * @returns {Integer}	Rescaled value 
	 */
	getValueToSendOverModbusTcp() {
		return this.point.getValueToSendOverModbusTcp();
	}
	
	/**
	* Gets current value multiplied by scale fixed to decimalPlaces, with replaced floatSeparator with unit (or true/false text or Enum Label) of the event field
	* @returns {String} 	Value of the event field
	*/
	getValueWithFacets() {
		return this.point.getValueWithFacets();
	}

	/**
	 * Sets decimal places
	 * @param {Integer} value 	Value between 0 and 3
	 * @returns {Boolean}		True if value was set, False if not.
	 */
	setDecimalPlaces(value) {
		return this.point.setDecimalPlaces(value);
	}

	/**
	 * Sets starting Modbus register address of Point
	 * @param {ModbusRequestContainer} modbusRequestContainer    	ModbusRequestContainer instance, where the register will be inited 
	 * @param {Integer} address										New starting Modbus register address
	 * @param {Object} objectBinding 								if referenceType is equal to "htmlId", then name (from XML) of HTML object binded to this register; if referenceType is equal to "object", then it is reference to collection object { "name": "", "value": "" }
	 * @param {String} dataType 									"int" or "unit" type provided as a string value
	 * @param {String} bitNumber 									"all" string value for numeric values or number in range <0, 15> for boolean values
	 * @param {Function} callback 									callback function to invoke if value will be updated
	 * @param {String} referenceType 								"htmlId" or "object" type provided as a string value
	 * @returns {Boolean} 											True if new address is in the range of AAC20 Modbus register addresses; False if new address is out of range AAC20 Modbus register addresses
	 */
	setModbusRegisterAddress(modbusRequestContainer, address, objectBinding, dataType = "uint", bitNumber = "all", callback = null, referenceType = "htmlId") {
		this.point.setModbusRegisterAddress(modbusRequestContainer, address, objectBinding, dataType, bitNumber, callback, referenceType);
	}

	/**
	 * Sets new type info for point
	 * @param {String} type 	New type - it mus be equal to "numeric", "boolean", "enum" or "unknown"
	 * @returns {Boolean} 		True if type was "numeric", "boolean", "enum" or "unknown"; False in other cases; 
	 */
	setPointType(type) {
		return this.point.setPointType(type);
	}
	
	/**
	 * Sets state - it is saved in current state value
	 * @param {Boolean} value 	Value to set
	 */
	setState(value) {
		this.point.setState(value);
	}
	
	/**
	 * Sets text to the HTML objcet related to instance of ActionButton
	 * @param {String} value 	Text to set
	 */
	setTextValue(value=null) {
		this.setLabelText((value != null) ? value : this.point.getValueWithFacets());
	}
	
	/**
	 * Sets value - it is saved in current value - this method adjusts current value to the number of decimal places
	 * @param {Float} value 	Value to set
	 */
	setValue(value) {
		this.point.setValue(value);
	}
}